/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */

package net.minecraftforge.gradle.userdev.tasks;

import com.google.common.collect.ImmutableMap;
import com.google.common.collect.ImmutableMultimap;
import net.minecraftforge.gradle.common.tasks.JarExec;
import net.minecraftforge.gradle.common.util.Utils;
import net.minecraftforge.srgutils.IMappingFile;
import org.gradle.api.file.ConfigurableFileCollection;
import org.gradle.api.file.RegularFile;
import org.gradle.api.file.RegularFileProperty;
import org.gradle.api.provider.Provider;
import org.gradle.api.tasks.InputFile;
import org.gradle.api.tasks.InputFiles;
import org.gradle.api.tasks.Internal;
import org.gradle.api.tasks.Optional;
import org.gradle.api.tasks.OutputFile;
import org.gradle.api.tasks.TaskAction;

import java.io.File;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardOpenOption;
import java.util.Arrays;
import java.util.List;

public abstract class RenameJar extends JarExec {
    private final Provider<RegularFile> tempMappings = this.workDir.map(s -> s.file("mappings.tsrg"));

    private boolean useArgsFile = false;

    public RenameJar() {
        getTool().set(Utils.FART);
        getArgs().addAll("--input", "{input}", "--output", "{output}", "--names", "{mappings}", "--lib", "{libraries}");
        getArgsFile().convention(workDir.map(d -> d.file("command_line_args.txt")));
    }

    @Override
    protected List<String> filterArgs(List<String> args) throws IOException {
        List<String> ret = replaceArgsMulti(args, ImmutableMap.of(
                        "{input}", getInput().get().getAsFile(),
                        "{output}", getOutput().get().getAsFile(),
                        "{mappings}", this.tempMappings.get().getAsFile()),
                ImmutableMultimap.<String, Object>builder()
                        .putAll("{libraries}", getLibraries().getFiles())
                        .build()
        );

        if (!useArgsFile)
            return ret;

        File argsFile = getArgsFile().get().getAsFile();
        Files.write(argsFile.toPath(), ret, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.TRUNCATE_EXISTING);
        return Arrays.asList("--cfg", argsFile.getAbsolutePath());
    }

    @TaskAction
    @Override
    public void apply() throws IOException {
        File tempMappings = this.tempMappings.get().getAsFile();

        if (tempMappings.getParentFile() != null && !tempMappings.getParentFile().exists() && !tempMappings.getParentFile().mkdirs())
            getProject().getLogger().warn("Could not create parent directories for temp dir '{}'", tempMappings.getAbsolutePath());

        if (tempMappings.exists() && !tempMappings.delete())
            throw new IllegalStateException("Could not delete temp mappings file: " + tempMappings.getAbsolutePath());

        IMappingFile mappings = IMappingFile.load(getMappings().get().getAsFile());

        for (File file : getExtraMappings().getFiles()) {
            mappings = mappings.merge(IMappingFile.load(file));
        }

        mappings.write(tempMappings.toPath(), IMappingFile.Format.TSRG2, false);

        super.apply();

        tempMappings.delete();
    }


    // TODO: Make this a ConfigurableFileCollection? (then remove getExtraMappings())
    @InputFile
    public abstract RegularFileProperty getMappings();

    @Optional
    @InputFiles
    public abstract ConfigurableFileCollection getExtraMappings();

    /**
     * The libraries to use for inheritance data during the renaming process.
     */
    @Optional
    @InputFiles
    public abstract ConfigurableFileCollection getLibraries();

    @InputFile
    public abstract RegularFileProperty getInput();

    @OutputFile
    public abstract RegularFileProperty getOutput();

    @Internal
    public abstract RegularFileProperty getArgsFile();

    @Internal
    public boolean getUseArgsFile() {
        return this.useArgsFile;
    }

    public void setUseArgsFile(boolean value) {
        this.useArgsFile = value;
    }

    public void useArgsFile() {
        this.useArgsFile = true;
    }
}
