/*
 * Decompiled with CFR 0.152.
 */
package net.minecraftforge.gitver.internal;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import net.minecraftforge.gitver.api.GitVersion;
import net.minecraftforge.gitver.api.GitVersionConfig;
import net.minecraftforge.gitver.api.GitVersionException;
import net.minecraftforge.gitver.internal.CommitCountProvider;
import net.minecraftforge.gitver.internal.GitChangelog;
import net.minecraftforge.gitver.internal.GitUtils;
import net.minecraftforge.gitver.internal.GitVersionExceptionInternal;
import net.minecraftforge.gitver.internal.GitVersionInternal;
import net.minecraftforge.gitver.internal.Lazy;
import net.minecraftforge.gitver.internal.Util;
import org.eclipse.jgit.api.Git;
import org.eclipse.jgit.api.errors.GitAPIException;
import org.eclipse.jgit.api.errors.RefNotFoundException;
import org.eclipse.jgit.lib.ObjectId;
import org.eclipse.jgit.lib.Ref;
import org.eclipse.jgit.lib.Repository;
import org.eclipse.jgit.revwalk.RevCommit;
import org.eclipse.jgit.util.StringUtils;
import org.jetbrains.annotations.Nullable;
import org.jetbrains.annotations.Unmodifiable;

public final class GitVersionImpl
implements GitVersionInternal {
    private final boolean strict;
    private Git git;
    private final Lazy<Info> info = Lazy.of(() -> this.calculateInfo(this::getSubprojectCommitCount));
    private final Lazy<String> url = Lazy.of(this::calculateUrl);
    private boolean closed = false;
    public final File gitDir;
    public final File root;
    public final File project;
    public final String localPath;
    private final List<File> includes;
    private final List<File> excludes;
    private final String tagPrefix;
    private final List<String> filters;
    private final List<File> subprojects;
    private final List<String> includesPaths;
    private final List<String> excludesPaths;
    private final List<String> subprojectPaths;
    private final Set<String> allIncludingPaths;
    private final Set<String> allExcludingPaths;

    public GitVersionImpl(File gitDir, File root, File project, GitVersionConfig config, boolean strict) {
        this.strict = strict;
        this.gitDir = gitDir;
        this.root = root;
        if (!this.gitDir.exists()) {
            throw new GitVersionExceptionInternal("Root directory is not a git repository");
        }
        this.project = project;
        if (this.project.compareTo(this.root) < 0) {
            throw new IllegalArgumentException("Project directory must be (a subdirectory of) the root directory");
        }
        try {
            config.validate(this.root);
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Invalid configuration", e);
        }
        this.localPath = GitVersionInternal.super.getProjectPath();
        GitVersionConfig.Project projectConfig = Objects.requireNonNull(config.getProject(this.localPath));
        this.includes = this.parsePaths(Arrays.asList(projectConfig.getIncludePaths()), false);
        this.includesPaths = this.makePaths(this.includes);
        this.excludes = this.parsePaths(Arrays.asList(projectConfig.getExcludePaths()), true);
        this.excludesPaths = this.makePaths(this.excludes);
        this.tagPrefix = this.makeTagPrefix(projectConfig.getTagPrefix());
        this.filters = this.makeFilters(projectConfig.getFilters());
        this.subprojects = this.parsePaths(config.getAllProjects(), GitVersionConfig.Project::getPath, true);
        this.subprojectPaths = this.makePaths(this.subprojects);
        HashSet<String> allIncludingPaths = new HashSet<String>(this.includesPaths);
        if (!this.localPath.isEmpty()) {
            allIncludingPaths.add(this.localPath);
        }
        this.allIncludingPaths = allIncludingPaths;
        HashSet<String> allExcludingPaths = new HashSet<String>(this.excludesPaths);
        allExcludingPaths.addAll(this.subprojectPaths);
        this.allExcludingPaths = allExcludingPaths;
    }

    @Override
    public String generateChangelog(@Nullable String start, @Nullable String url, boolean plainText) throws GitVersionException {
        try {
            RevCommit from;
            this.open();
            if (StringUtils.isEmptyOrNull((String)start)) {
                RevCommit mergeBase = GitUtils.getMergeBaseCommit(this.git);
                from = mergeBase != null ? mergeBase : GitUtils.getFirstCommitInRepository(this.git);
            } else {
                Map<String, String> tags = GitUtils.getTagToCommitMap(this.git);
                String commitHash = Util.orElse(tags.get(start), () -> start);
                from = GitUtils.getCommitFromId(this.git, ObjectId.fromString((String)commitHash));
            }
            if (from == null) {
                throw new GitVersionExceptionInternal("Opened repository has no commits");
            }
            RevCommit head = GitUtils.getHead(this.git);
            return GitChangelog.generateChangelogFromTo(this.git, Util.orElse(url, () -> GitUtils.buildProjectUrl(this.git)), plainText, from, head, this.tagPrefix, List.of());
        }
        catch (IOException | GitVersionException | GitAPIException e) {
            if (this.strict) {
                throw new GitVersionExceptionInternal("Failed to generate the changelog", e);
            }
            return "";
        }
    }

    @Override
    public GitVersion.Info getInfo() {
        return this.info.get();
    }

    @Override
    @Nullable
    public String getUrl() {
        return this.url.get();
    }

    private Info calculateInfo(CommitCountProvider commitCountProvider) {
        try {
            this.open();
            String describedTag = Util.make(this.git.describe(), it -> {
                it.setTags(true);
                it.setLong(true);
                try {
                    it.setMatch(new String[]{this.tagPrefix + "[[:digit:]]**"});
                    for (String filter : this.filters) {
                        if (filter.startsWith("!")) {
                            it.setExclude(new String[]{filter.substring(1)});
                            continue;
                        }
                        it.setMatch(new String[]{filter});
                    }
                }
                catch (Exception e) {
                    Util.sneak(e);
                }
            }).call();
            if (describedTag == null) {
                throw new RefNotFoundException("Tag not found! A valid tag must include a digit! Tag prefix: %s, Filters: %s".formatted(!this.tagPrefix.isEmpty() ? this.tagPrefix : "NONE!", String.join((CharSequence)", ", this.filters)));
            }
            String[] desc = Util.rsplit(describedTag, "-", 2);
            Ref head = this.git.getRepository().exactRef("HEAD");
            String longBranch = Util.make(() -> {
                if (!head.isSymbolic()) {
                    return null;
                }
                Ref target = head.getTarget();
                return target != null ? target.getName() : null;
            });
            String tag = desc[0].substring(Util.indexOf(desc[0], Character::isDigit, 0));
            String offset = commitCountProvider.getAsString(this.git, desc[0], desc[1], this.strict);
            String hash = desc[2];
            String branch = longBranch != null ? Repository.shortenRefName((String)longBranch) : null;
            String commit = ObjectId.toString((ObjectId)head.getObjectId());
            String abbreviatedId = head.getObjectId().abbreviate(8).name();
            return new Info(tag, offset, hash, branch, commit, abbreviatedId);
        }
        catch (Exception e) {
            if (this.strict) {
                throw new GitVersionExceptionInternal("Failed to calculate version info", e);
            }
            return Info.EMPTY;
        }
    }

    @Nullable
    private String calculateUrl() {
        try {
            this.open();
            return GitUtils.buildProjectUrl(this.git);
        }
        catch (Exception e) {
            return null;
        }
    }

    @Override
    public String getTagPrefix() {
        return this.tagPrefix;
    }

    private String makeTagPrefix(String tagPrefix) {
        if (StringUtils.isEmptyOrNull((String)tagPrefix) || tagPrefix.isBlank()) {
            return "";
        }
        return !tagPrefix.endsWith("-") ? tagPrefix + "-" : tagPrefix;
    }

    @Override
    public @Unmodifiable Collection<String> getFilters() {
        return this.filters;
    }

    private @Unmodifiable List<String> makeFilters(String ... filters) {
        ArrayList<String> list = new ArrayList<String>(filters.length);
        for (String s : filters) {
            if (s.length() <= (s.startsWith("!") ? 1 : 0)) continue;
            list.add(s);
        }
        return Collections.unmodifiableList(list);
    }

    @Override
    public File getGitDir() {
        return this.gitDir;
    }

    @Override
    public File getRoot() {
        return this.root;
    }

    @Override
    public File getProject() {
        return this.project;
    }

    @Override
    public String getProjectPath() {
        return this.localPath;
    }

    private @Unmodifiable List<File> parsePaths(Collection<String> paths, boolean removeParents) {
        return this.parsePaths(paths, Function.identity(), removeParents);
    }

    private <T> @Unmodifiable List<File> parsePaths(Collection<T> paths, Function<T, String> mapper, boolean removeParents) {
        ArrayList<File> ret = new ArrayList<File>(paths.size());
        for (T o : paths) {
            String path;
            String p = mapper.apply(o);
            File file = new File(this.root, p).getAbsoluteFile();
            if (removeParents && ((path = GitUtils.getRelativePath(this.project, file)).isEmpty() || path.startsWith("../") || path.equals("..") || path.equals("."))) continue;
            ret.add(file);
        }
        return Collections.unmodifiableList(ret);
    }

    private @Unmodifiable List<String> makePaths(Collection<? extends File> files) {
        ArrayList<String> ret = new ArrayList<String>(files.size());
        for (File file : files) {
            String path = GitUtils.getRelativePath(this.getRoot(), file);
            if (path.isBlank()) continue;
            ret.add(path);
        }
        return Collections.unmodifiableList(ret);
    }

    @Override
    public @Unmodifiable Collection<File> getIncludes() {
        return this.includes;
    }

    @Override
    public @Unmodifiable Collection<File> getExcludes() {
        return this.excludes;
    }

    @Override
    public @Unmodifiable Collection<String> getIncludesPaths() {
        return this.includesPaths;
    }

    @Override
    public @Unmodifiable Collection<String> getExcludesPaths() {
        return this.excludesPaths;
    }

    @Override
    public @Unmodifiable Collection<File> getSubprojects() {
        return this.subprojects;
    }

    private int getSubprojectCommitCount(Git git, String tag) {
        if (this.localPath.isEmpty() && this.allExcludingPaths.isEmpty()) {
            return -1;
        }
        try {
            int count = GitUtils.countCommits(git, tag, this.tagPrefix, this.allIncludingPaths, this.allExcludingPaths);
            return Math.max(count, 0);
        }
        catch (IOException | GitAPIException e) {
            throw new GitVersionExceptionInternal("Failed to count commits with the following parameters: Tag %s, Include Paths [%s], Exclude Paths [%s]".formatted(tag, String.join((CharSequence)", ", this.allIncludingPaths), String.join((CharSequence)", ", this.allExcludingPaths)));
        }
    }

    @Override
    public @Unmodifiable Collection<String> getSubprojectPaths() {
        return this.subprojectPaths;
    }

    private void open() {
        if (this.git != null) {
            return;
        }
        if (this.closed) {
            throw new GitVersionExceptionInternal("GitVersion is closed!");
        }
        try {
            this.git = Git.open((File)this.gitDir);
        }
        catch (IOException e) {
            this.close();
            throw new GitVersionExceptionInternal("Failed to open Git repository", e);
        }
    }

    @Override
    public void close() {
        this.closed = true;
        if (this.git == null) {
            return;
        }
        this.git.close();
        this.git = null;
    }

    public static GitVersion empty(@Nullable File project) {
        return new Empty(project);
    }

    public record Info(String getTag, String getOffset, String getHash, String getBranch, String getCommit, String getAbbreviatedId) implements GitVersionInternal.Info
    {
        private static final Info EMPTY = new Info("0.0", "0", "00000000", "master", "0000000000000000000000", "00000000");
    }

    public record Empty(@Nullable File project) implements GitVersionInternal
    {
        @Override
        public String generateChangelog(@Nullable String start, @Nullable String url, boolean plainText) throws GitVersionException {
            throw new GitVersionExceptionInternal("Cannot generate a changelog without a repository");
        }

        @Override
        public GitVersionInternal.Info getInfo() throws GitVersionException {
            return Info.EMPTY;
        }

        @Override
        @Nullable
        public String getUrl() {
            return null;
        }

        @Override
        public String getTagPrefix() {
            throw new GitVersionExceptionInternal("Cannot get tag prefix from an empty repository");
        }

        @Override
        public @Unmodifiable Collection<String> getFilters() {
            throw new GitVersionExceptionInternal("Cannot get filters from an empty repository");
        }

        @Override
        public File getGitDir() {
            throw new GitVersionExceptionInternal("Cannot get git directory from an empty repository");
        }

        @Override
        public File getRoot() {
            throw new GitVersionExceptionInternal("Cannot get root directory from an empty repository");
        }

        @Override
        public File getProject() {
            if (this.project != null) {
                return this.project;
            }
            throw new GitVersionExceptionInternal("Cannot get project directory without a project");
        }

        @Override
        public @Unmodifiable Collection<File> getIncludes() {
            throw new GitVersionExceptionInternal("Cannot get include paths from an empty repository");
        }

        @Override
        public @Unmodifiable Collection<File> getExcludes() {
            throw new GitVersionExceptionInternal("Cannot get exclude paths from an empty repository");
        }

        @Override
        public @Unmodifiable Collection<String> getIncludesPaths() {
            throw new GitVersionExceptionInternal("Cannot get include paths from an empty repository");
        }

        @Override
        public @Unmodifiable Collection<String> getExcludesPaths() {
            throw new GitVersionExceptionInternal("Cannot get exclude paths from an empty repository");
        }

        @Override
        public @Unmodifiable Collection<File> getSubprojects() {
            throw new GitVersionExceptionInternal("Cannot get subprojects from an empty repository");
        }

        @Override
        public Collection<String> getSubprojectPaths() {
            throw new GitVersionExceptionInternal("Cannot get subprojects from an empty repository");
        }

        @Override
        public void close() {
        }
    }
}

