/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */
package net.minecraftforge.gradle;

import org.gradle.api.artifacts.ExternalModuleDependency;
import org.gradle.api.provider.Provider;
import org.gradle.api.provider.ProviderConvertible;

import java.io.File;

/// A Mavenizer instance is an abstraction over an invocation of the Minecraft Mavenizer and its output when generating
/// a Minecraft dependency. This is designed to be consumed by one of the various
/// [MinecraftExtensionForProject#dependency] methods as it is a [ProviderConvertible] of a dependency.
///
/// This is not to be confused with [MinecraftDependency], which is designed to add additional DSL features on top of
/// [ExternalModuleDependency] when configuring a Minecraft dependency in your buildscript.
public interface MavenizerInstance extends ProviderConvertible<ExternalModuleDependency> {
    /// Gets the dependency (used by Gradle) generated by the Mavenizer instance.
    ///
    /// @return The dependency generated by this instance
    @Override Provider<ExternalModuleDependency> asProvider();

    /// Gets the mappings version used by the Mavenizer instance.
    ///
    /// @return The mappings version used
    /// @see MinecraftMappings#getVersion()
    Provider<String> getMappingVersion();

    /// Gets the artifact coordinates for the SRG mappings used by the Mavenizer instance.
    ///
    /// @return The SRG mapping artifact coordinate
    Provider<String> getToSrg();

    /// Gets the SRG mapping file used by the Mavenizer instance.
    ///
    /// @return The SRG mapping file
    /// @apiNote Buildscript authors should prefer to use [#getToSrg()] with
    /// [org.gradle.api.artifacts.dsl.DependencyFactory#create(CharSequence)] or
    /// [org.gradle.api.artifacts.dsl.DependencyHandler#addProvider(String, Provider)].
    Provider<File> getToSrgFile();

    /// Gets the artifact coordinates for the SRG mappings used by the Mavenizer instance.
    ///
    /// @return The SRG mapping artifact coordinate
    Provider<String> getToObf();

    /// Gets the notch mapping file used by the Mavenizer instance.
    ///
    /// @return The notch mapping file
    /// @apiNote Buildscript authors should prefer to use [#getToObf()] with
    /// [org.gradle.api.artifacts.dsl.DependencyFactory#create(CharSequence)] or
    /// [org.gradle.api.artifacts.dsl.DependencyHandler#addProvider(String, Provider)].
    Provider<File> getToObfFile();
}
