/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */
package net.minecraftforge.gradle;

import org.gradle.api.Named;
import org.gradle.api.NamedDomainObjectContainer;
import org.gradle.api.file.Directory;
import org.gradle.api.file.DirectoryProperty;
import org.gradle.api.file.FileCollection;
import org.gradle.api.file.ProjectLayout;
import org.gradle.api.model.ObjectFactory;
import org.gradle.api.provider.ListProperty;
import org.gradle.api.provider.MapProperty;
import org.gradle.api.provider.Property;
import org.gradle.api.provider.Provider;
import org.gradle.api.tasks.Classpath;
import org.gradle.api.tasks.Input;
import org.gradle.api.tasks.InputDirectory;
import org.gradle.api.tasks.InputFiles;
import org.gradle.api.tasks.Optional;
import org.jetbrains.annotations.Nullable;

import javax.inject.Inject;
import java.io.File;
import java.util.Arrays;
import java.util.Map;

/// The configuration options for Slime Launcher tasks.
///
/// The launch tasks generated by the [Minecraft][MinecraftExtension] extension are specialized
/// [JavaExec][org.gradle.api.tasks.JavaExec] tasks that are designed to work with Slime Launcher, Minecraft Forge's
/// dedicated launcher for the development environment. While the implementing task class remains internal (can still be
/// configured as type `JavaExec`), these options exist to allow consumers to alter or add pre-defined attributes to the
/// run configurations as needed.
///
/// For example, changing the [main class][org.gradle.api.tasks.JavaExec#getMainClass()] of the launch task as a
/// `JavaExec` task will cause it to not use Slime Launcher and skip its configurations for it. If a consumer wishes to
/// use Slime Launcher but change the main class it delegates to after initial setup, that can be done using
/// [#setMainClass(String)] or [#setMainClass(Provider)].
///
/// @apiNote This class is public-facing as a class instead of an interface to satisfy the requirement that Gradle's
/// [org.gradle.api.NamedDomainObjectContainer] must house a class that implements the [Named] interface. Like the other
/// public-facing interface APIs in ForgeGradle, this class remains sealed and is implemented by a package-private class
/// that cannot be directly accessed.
@SuppressWarnings("unused") // TODO [ForgeGradle7][Testing] Write functional tests
public sealed abstract class SlimeLauncherOptions implements Named permits SlimeLauncherOptions.Impl {
    /// The name of the Slime Launcher configuration.
    ///
    /// This will be used to create the task name with the verb "run" using
    /// [org.gradle.api.tasks.SourceSet#getTaskName(String, String)]
    ///
    /// @return The name of this configuration
    @Override
    public final String getName() {
        return this.name;
    }

    /// The main class for Slime Launcher to use.
    ///
    /// This is the class that will be invoked by Slime Launcher, **not** the main class of the
    /// [org.gradle.api.tasks.JavaExec] task that will be produced from these options.
    ///
    /// @return A property for the main class
    public final @Input @Optional Property<String> getMainClass() {
        return this.mainClass;
    }

    /// The arguments to pass to the main class.
    ///
    /// This is the arguments that will be passed to the main class through Slime Launcher, **not** the arguments for
    /// Slime Launcher itself.
    ///
    /// @return A property for the arguments to pass to the main class
    public final @Input @Optional ListProperty<String> getArgs() {
        return this.args;
    }

    /// The JVM arguments to use.
    ///
    /// These are applied immediately when Slime Launcher is executed. A reminder that Slime Launcher is not a
    /// re-launcher but a dev-environment bootstrapper.
    ///
    /// @return A property for the JVM arguments
    public final @Input @Optional ListProperty<String> getJvmArgs() {
        return this.jvmArgs;
    }

    /// The classpath to use.
    ///
    /// The classpath in question **must include** Slime Launcher in it. By default, the [minecraft][MinecraftExtension]
    /// extension adds Slime Launcher as a dependency to the consuming [project's][org.gradle.api.Project]
    /// [runtimeClasspath][org.gradle.api.plugins.JavaPlugin#RUNTIME_CLASSPATH_CONFIGURATION_NAME] configuration.
    ///
    /// Keep in mind that if the [org.gradle.api.tasks.JavaExec] task is configured to have a different [main
    /// class][org.gradle.api.tasks.JavaExec#getMainClass()], the classpath does not need to include Slime Launcher.
    ///
    /// @return The classpath to use
    public final @InputFiles @Optional @Classpath FileCollection getClasspath() {
        return this.classpath == null ? this.classpath = this.getObjects().fileCollection() : this.classpath;
    }

    /// The minimum memory heap size to use.
    ///
    /// Working with this property is preferred over manually using the `-Xms` argument in the [JVM
    /// arguments][#getJvmArgs()].
    ///
    /// @return A property for the minimum heap size
    public final @Input @Optional Property<String> getMinHeapSize() {
        return this.minHeapSize;
    }

    /// The maximum memory heap size to use.
    ///
    /// Working with this property is preferred over manually using the `-Xmx` argument in the [JVM
    /// arguments][#getJvmArgs()].
    ///
    /// @return A property for the maximum heap size
    public final @Input @Optional Property<String> getMaxHeapSize() {
        return this.maxHeapSize;
    }

    /// The system properties to use.
    ///
    /// @return A property for the system properties
    public final @Input @Optional MapProperty<String, Object> getSystemProperties() {
        return this.systemProperties;
    }

    /// The environment variables to use.
    ///
    /// @return A property for the environment variables
    public final @Input @Optional MapProperty<String, Object> getEnvironment() {
        return this.environment;
    }

    /// The working directory to use.
    ///
    /// By default, this will be `run/`{@link #getName() name}.
    ///
    /// To clarify: this is the working directory of the Java process. Slime Launcher uses absolute file locations to
    /// place its caches and metadata, which do not interfere with the working directory.
    ///
    /// @return A property for the working directory
    public final @InputDirectory DirectoryProperty getWorkingDir() {
        return this.workingDir;
    }

    /// Sets the main class for Slime Launcher to use.
    ///
    /// @param mainClass The main class
    /// @see #getMainClass()
    public final void setMainClass(String mainClass) {
        this.getMainClass().set(mainClass);
    }

    /// Sets the main class for Slime Launcher to use.
    ///
    /// @param mainClass The main class
    /// @see #getMainClass()
    public final void setMainClass(Provider<String> mainClass) {
        this.getMainClass().set(mainClass);
    }

    /// Adds to the arguments to pass to the [main class][#getMainClass()].
    ///
    /// @param args The arguments to add
    /// @apiNote To add multiple arguments, use [#args(String...)]
    /// @see #getArgs()
    public final void args(String args) {
        this.getArgs().add(args);
    }

    /// Adds to the arguments to pass to the [main class][#getMainClass()].
    ///
    /// @param args The arguments to add
    /// @apiNote Unlike [#setArgs(String...)], this method does not replace the existing arguments.
    /// @see #getArgs()
    public final void args(String... args) {
        this.getArgs().addAll(args);
    }

    /// Adds to the arguments to pass to the [main class][#getMainClass()].
    ///
    /// @param args The arguments to add
    /// @apiNote Unlike [#setArgs(Iterable)], this method does not replace the existing arguments.
    /// @see #getArgs()
    public final void args(Iterable<String> args) {
        this.getArgs().addAll(args);
    }

    /// Adds to the arguments to pass to the [main class][#getMainClass()].
    ///
    /// @param args The arguments to add
    /// @apiNote Unlike [#setArgs(Provider)], this method does not replace the existing arguments.
    /// @see #getArgs()
    public final void args(Provider<? extends Iterable<String>> args) {
        this.getArgs().addAll(args);
    }

    /// Sets the arguments to pass to the [main class][#getMainClass()].
    ///
    /// @param args The arguments
    /// @apiNote This method will replace any existing arguments. To add to existing arguments, use [#args(String...)].
    /// @see #getArgs()
    public final void setArgs(String... args) {
        this.getArgs().set(Arrays.asList(args));
    }

    /// Sets the arguments to pass to the [main class][#getMainClass()].
    ///
    /// @param args The arguments
    /// @apiNote This method will replace any existing arguments. To add to existing arguments, use [#args(Iterable)].
    /// @see #getArgs()
    public final void setArgs(Iterable<String> args) {
        this.getArgs().set(args);
    }

    /// Sets the arguments to pass to the [main class][#getMainClass()].
    ///
    /// @param args The arguments
    /// @apiNote This method will replace any existing arguments. To add to existing arguments, use [#args(Provider)].
    /// @see #getArgs()
    public final void setArgs(Provider<? extends Iterable<String>> args) {
        this.getArgs().set(args);
    }

    /// Adds to the JVM arguments to use.
    ///
    /// @param jvmArgs The JVM argument to add
    /// @apiNote To add multiple arguments, use [#jvmArgs(String...)]
    /// @see #getJvmArgs()
    public final void jvmArgs(String jvmArgs) {
        this.getJvmArgs().add(jvmArgs);
    }

    /// Adds to the JVM arguments to use.
    ///
    /// @param jvmArgs The JVM arguments to add
    /// @apiNote Unlike [#setJvmArgs(String...)], this method does not replace the existing arguments.
    /// @see #getJvmArgs()
    public final void jvmArgs(String... jvmArgs) {
        this.getJvmArgs().addAll(jvmArgs);
    }

    /// Adds to the JVM arguments to use.
    ///
    /// @param jvmArgs The JVM arguments to add
    /// @apiNote Unlike [#setJvmArgs(Iterable)], this method does not replace the existing arguments.
    /// @see #getJvmArgs()
    public final void jvmArgs(Iterable<String> jvmArgs) {
        this.getJvmArgs().addAll(jvmArgs);
    }

    /// Adds to the JVM arguments to use.
    ///
    /// @param jvmArgs The JVM arguments to add
    /// @apiNote Unlike [#setJvmArgs(Provider)], this method does not replace the existing arguments.
    /// @see #getJvmArgs()
    public final void jvmArgs(Provider<? extends Iterable<String>> jvmArgs) {
        this.getJvmArgs().addAll(jvmArgs);
    }

    /// Sets the JVM arguments to use.
    ///
    /// @param jvmArgs The arguments
    /// @apiNote This method will replace any existing arguments. To add to existing arguments, use
    /// [#jvmArgs(String...)].
    /// @see #getJvmArgs()
    public final void setJvmArgs(String... jvmArgs) {
        this.getJvmArgs().set(Arrays.asList(jvmArgs));
    }

    /// Sets the JVM arguments to use.
    ///
    /// @param jvmArgs The arguments
    /// @apiNote This method will replace any existing arguments. To add to existing arguments, use
    /// [#jvmArgs(Iterable)].
    /// @see #getJvmArgs()
    public final void setJvmArgs(Iterable<String> jvmArgs) {
        this.getJvmArgs().set(jvmArgs);
    }

    /// Sets the JVM arguments to use.
    ///
    /// @param jvmArgs The arguments
    /// @apiNote This method will replace any existing arguments. To add to existing arguments, use
    /// [#jvmArgs(Provider)].
    /// @see #getJvmArgs()
    public final void setJvmArgs(Provider<? extends Iterable<String>> jvmArgs) {
        this.getJvmArgs().set(jvmArgs);
    }

    /// Adds to the classpath to use.
    ///
    /// @param classpath The classpath to include with the existing classpath
    /// @apiNote Unlike [#setClasspath(Object...)], this method does not replace the existing classpath.
    /// @see #getClasspath()
    public final void classpath(Object... classpath) {
        this.classpath(this.getObjects().fileCollection().from(classpath));
    }

    /// Adds to the classpath to use.
    ///
    /// @param classpath The classpath to include with the existing classpath
    /// @apiNote Unlike [#setClasspath(Iterable)], this method does not replace the existing classpath.
    /// @see #getClasspath()
    public final void classpath(Iterable<?> classpath) {
        this.classpath(this.getObjects().fileCollection().from(classpath));
    }

    /// Adds to the classpath to use.
    ///
    /// @param classpath The classpath to include with the existing classpath
    /// @apiNote Unlike [#setClasspath(Provider)], this method does not replace the existing classpath.
    /// @see #getClasspath()
    public final void classpath(Provider<? extends Iterable<?>> classpath) {
        this.classpath(this.getObjects().fileCollection().from(classpath));
    }

    /// Adds to the classpath to use.
    ///
    /// @param classpath The classpath to include with the existing classpath
    /// @apiNote Unlike [#setClasspath(FileCollection)], this method does not replace the existing classpath.
    /// @see #getClasspath()
    public final void classpath(FileCollection classpath) {
        this.getClasspath().plus(classpath);
    }

    /// Sets the classpath to use.
    ///
    /// @param classpath The classpath
    /// @apiNote This method will replace the existing classpath. To add to it, use [#classpath(Object...)].
    /// @see #getJvmArgs()
    public final void setClasspath(Object... classpath) {
        this.setClasspath(this.getObjects().fileCollection().from(classpath));
    }

    /// Sets the classpath to use.
    ///
    /// @param classpath The classpath
    /// @apiNote This method will replace the existing classpath. To add to it, use [#classpath(Iterable)].
    /// @see #getJvmArgs()
    public final void setClasspath(Iterable<?> classpath) {
        this.setClasspath(this.getObjects().fileCollection().from(classpath));
    }

    /// Sets the classpath to use.
    ///
    /// @param classpath The classpath
    /// @apiNote This method will replace the existing classpath. To add to it, use [#classpath(Provider)].
    /// @see #getJvmArgs()
    public final void setClasspath(Provider<? extends Iterable<?>> classpath) {
        this.setClasspath(this.getObjects().fileCollection().from(classpath.get()));
    }

    /// Sets the classpath to use.
    ///
    /// @param classpath The classpath
    /// @apiNote This method will replace the existing classpath. To add to it, use [#classpath(FileCollection)].
    /// @see #getJvmArgs()
    public final void setClasspath(FileCollection classpath) {
        this.classpath = classpath;
    }

    /// Sets the minimum memory heap size to use.
    ///
    /// @param minHeapSize The minimum heap size
    /// @see #getMinHeapSize()
    public final void setMinHeapSize(String minHeapSize) {
        this.getMinHeapSize().set(minHeapSize);
    }

    /// Sets the minimum memory heap size to use.
    ///
    /// @param minHeapSize The minimum heap size
    /// @see #getMinHeapSize()
    public final void setMinHeapSize(Property<String> minHeapSize) {
        this.getMinHeapSize().set(minHeapSize);
    }

    /// Sets the maximum memory heap size to use.
    ///
    /// @param maxHeapSize The maximum heap size
    /// @see #getMaxHeapSize()
    public final void setMaxHeapSize(String maxHeapSize) {
        this.getMaxHeapSize().set(maxHeapSize);
    }

    /// Sets the maximum memory heap size to use.
    ///
    /// @param maxHeapSize The maximum heap size
    /// @see #getMaxHeapSize()
    public final void setMaxHeapSize(Property<String> maxHeapSize) {
        this.getMaxHeapSize().set(maxHeapSize);
    }

    /// Adds a single system property to use.
    ///
    /// @param name  The name
    /// @param value The value
    /// @apiNote To add multiple system properties at once, use [#systemProperties(Provider)].
    /// @see #getSystemProperties()
    public final void systemProperty(String name, Object value) {
        this.getSystemProperties().put(name, value);
    }

    /// Adds to the system properties to use.
    ///
    /// @param properties The system properties
    /// @apiNote Unlike [#setSystemProperties(Map)], this method does not replace the existing system properties. To add
    /// a single property, use [#systemProperty(String,Object)].
    /// @see #getSystemProperties()
    public final void systemProperties(Map<String, ?> properties) {
        this.getSystemProperties().putAll(properties);
    }

    /// Adds to the system properties to use.
    ///
    /// @param properties The system properties
    /// @apiNote Unlike [#setSystemProperties(Provider)], this method does not replace the existing system properties.
    /// To add a single property, use [#systemProperty(String,Object)].
    /// @see #getSystemProperties()
    public final void systemProperties(Provider<? extends Map<String, ?>> properties) {
        this.getSystemProperties().putAll(properties);
    }

    /// Sets the system properties to use.
    ///
    /// @param properties The system properties
    /// @apiNote This method will replace any existing system properties. To add to them, use [#systemProperties(Map)]
    /// or [#systemProperty(String,Object)].
    /// @see #getSystemProperties()
    public final void setSystemProperties(Map<String, ?> properties) {
        this.getSystemProperties().set(properties);
    }

    /// Sets the system properties to use.
    ///
    /// @param properties The system properties
    /// @apiNote This method will replace any existing system properties. To add to them, use
    /// [#systemProperties(Provider)] or [#systemProperty(String,Object)].
    /// @see #getSystemProperties()
    public final void setSystemProperties(Provider<? extends Map<String, ?>> properties) {
        this.getSystemProperties().set(properties);
    }

    /// Adds a single environment variable to use.
    ///
    /// @param name  The name
    /// @param value The value
    /// @apiNote To add multiple environment variables at once, use [#environment(Provider)].
    /// @see #getEnvironment()
    public final void environment(String name, Object value) {
        this.getEnvironment().put(name, value);
    }

    /// Adds to the environment variables to use.
    ///
    /// @param properties The environment variables
    /// @apiNote Unlike [#setEnvironment(Map)], this method does not replace the existing environment variables. To add
    /// a single variable, use [#environment(String,Object)].
    /// @see #getEnvironment()
    public final void environment(Map<String, ?> properties) {
        this.getEnvironment().putAll(properties);
    }

    /// Adds to the environment variables to use.
    ///
    /// @param properties The environment variables
    /// @apiNote Unlike [#setEnvironment(Provider)], this method does not replace the existing environment variables. To
    /// add a single variable, use [#environment(String,Object)].
    /// @see #getEnvironment()
    public final void environment(Provider<? extends Map<String, ?>> properties) {
        this.getEnvironment().putAll(properties);
    }

    /// Sets the environment variables to use.
    ///
    /// @param environment The environment variables
    /// @apiNote This method will replace any existing environment variables. To add to them, use [#environment(Map)] or
    /// [#environment(String,Object)].
    /// @see #getEnvironment()
    public final void setEnvironment(Map<String, ?> environment) {
        this.getEnvironment().set(environment);
    }

    /// Sets the environment variables to use.
    ///
    /// @param environment The environment variables
    /// @apiNote This method will replace any existing environment variables. To add to them, use
    /// [#environment(Provider)] or [#environment(String,Object)].
    /// @see #getEnvironment()
    public final void setEnvironment(Provider<? extends Map<String, ?>> environment) {
        this.getEnvironment().set(environment);
    }

    /// Sets the working directory to use.
    ///
    /// The given file must be a [directory][File#isDirectory()].
    ///
    /// @param workingDir The working directory
    /// @see #getWorkingDir()
    public final void setWorkingDir(File workingDir) {
        if (!workingDir.isDirectory())
            throw new IllegalStateException("The working directory must be a directory: " + workingDir);

        // Run the file through the resolver to validate that it's actually a directory
        this.getWorkingDir().fileValue(workingDir);
    }

    /// Sets the working directory to use.
    ///
    /// @param workingDir The working directory
    /// @see #getWorkingDir()
    public final void setWorkingDir(Directory workingDir) {
        this.getWorkingDir().set(workingDir);
    }

    /// Sets the working directory to use.
    ///
    /// @param workingDir The working directory
    /// @apiNote Note that the provider given must hold a [Directory], not a standard [File]. If you must use a standard
    /// file object, use [#setWorkingDir(File)].
    /// @see #setWorkingDir(Directory)
    public final void setWorkingDir(Provider<? extends Directory> workingDir) {
        this.getWorkingDir().set(workingDir);
    }


    /* INTERNAL */

    private final String name;

    private final @Input @Optional Property<String> mainClass;
    private final @Input @Optional ListProperty<String> args;
    private final @Input @Optional ListProperty<String> jvmArgs;
    private @InputFiles @Optional @Classpath @Nullable FileCollection classpath;
    private final @Input @Optional Property<String> minHeapSize;
    private final @Input @Optional Property<String> maxHeapSize;
    private final @Input @Optional MapProperty<String, Object> systemProperties;
    private final @Input @Optional MapProperty<String, Object> environment;
    private final @InputDirectory DirectoryProperty workingDir;

    /// Creates a new Slime Launcher options object.
    ///
    /// @param name The name of the Slime Launcher configuration
    private SlimeLauncherOptions(String name) {
        this.name = name;

        this.mainClass = this.getObjects().property(String.class);
        this.args = this.getObjects().listProperty(String.class);
        this.jvmArgs = this.getObjects().listProperty(String.class);
        this.minHeapSize = this.getObjects().property(String.class);
        this.maxHeapSize = this.getObjects().property(String.class);
        this.systemProperties = this.getObjects().mapProperty(String.class, Object.class);
        this.environment = this.getObjects().mapProperty(String.class, Object.class);
        this.workingDir = this.getObjects().directoryProperty().convention(this.getLayout().getProjectDirectory().dir("run").dir(name));
    }

    abstract ObjectFactory getObjects();

    abstract ProjectLayout getLayout();

    abstract non-sealed static class Impl extends SlimeLauncherOptions {
        @Inject
        public Impl(String name) {
            super(name);
        }

        @Override
        protected abstract @Inject ObjectFactory getObjects();

        @Override
        protected abstract @Inject ProjectLayout getLayout();
    }

    /// Creates a Slime Launcher options named domain object container.
    ///
    /// @param objects The Gradle object factory
    /// @return The factory
    static NamedDomainObjectContainer<SlimeLauncherOptions> container(ObjectFactory objects, ProjectLayout layout) {
        return objects.domainObjectContainer(SlimeLauncherOptions.class, name -> objects.<SlimeLauncherOptions>newInstance(Impl.class, name));
    }

    /// Applies the options to the given task.
    ///
    /// @param task The task to apply the options to
    final void apply(SlimeLauncherExec task) {
        if (this.mainClass.map(Util::nullIfEmpty).isPresent())
            task.getBootstrapMainClass().set(this.mainClass);

        if (this.args.map(Util::nullIfEmpty).isPresent())
            task.getMcBootstrapArgs().set(this.args);

        if (this.jvmArgs.map(Util::nullIfEmpty).isPresent())
            task.jvmArgs(this.jvmArgs.get());

        if (this.classpath != null && !this.classpath.isEmpty())
            task.setClasspath(this.classpath);

        if (this.minHeapSize.map(Util::nullIfEmpty).isPresent())
            task.setMinHeapSize(this.minHeapSize.get());

        if (this.maxHeapSize.map(Util::nullIfEmpty).isPresent())
            task.setMinHeapSize(this.maxHeapSize.get());

        if (this.systemProperties.map(Util::nullIfEmpty).isPresent())
            task.systemProperties(this.systemProperties.get());

        if (this.environment.map(Util::nullIfEmpty).isPresent())
            task.environment(this.environment.get());

        if (this.workingDir.isPresent())
            task.workingDir(this.workingDir);
    }
}
