/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */
package net.minecraftforge.gradle;

import groovy.lang.Closure;
import groovy.lang.DelegatesTo;
import groovy.transform.Generated;
import groovy.transform.NamedParam;
import groovy.transform.NamedParams;
import groovy.transform.NamedVariant;
import groovy.transform.stc.ClosureParams;
import groovy.transform.stc.SimpleType;
import net.minecraftforge.accesstransformers.gradle.AccessTransformersContainer;
import org.gradle.api.Action;
import org.gradle.api.artifacts.ExternalModuleDependency;
import org.gradle.api.file.RegularFile;
import org.gradle.api.provider.Provider;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.Nullable;

import java.io.File;
import java.util.Map;

public sealed interface MinecraftDependency extends ExternalModuleDependency permits MinecraftDependencyInternal {
    @Nullable MinecraftExtension.Mappings getMappings();

    /**
     * Sets the mappings to use for the Minecraft Maven.
     * <p>This method includes a generated named variant that can make declaration in your buildscript easier.</p>
     * <pre><code>
     * minecraft {
     *     mappings channel: 'official', version: '1.21.5'
     * }
     * </code></pre>
     *
     * @param channel The mappings channel
     * @param version The mappings version
     * @throws IllegalArgumentException If any parameter is {@code null}
     * @apiNote Mappings should only be declared once. A warning will be reported on re-declaration.
     * @see <a href="https://docs.groovy-lang.org/latest/html/api/groovy/transform/NamedVariant.html">NamedVariant</a>
     */
    @NamedVariant
    void mappings(String channel, String version);

    /// Sets the mappings to use for the Minecraft Maven.
    ///
    /// This method is generated by Groovy in the implementing class, but has been included here for convenience and IDE
    /// support.
    ///
    /// @param namedArgs The named arguments
    /// @throws IllegalArgumentException If any parameter is `null`
    /// @apiNote Mappings should only be declared once. A warning will be reported on re-declaration.
    /// @see #mappings(String, String)
    @Contract // empty contract so IDEs don't think this method always fails
    @Generated
    @SuppressWarnings("rawtypes")
    default void mappings(
        @NamedParams({
            @NamedParam(
                type = String.class,
                value = "channel",
                required = true
            ),
            @NamedParam(
                type = String.class,
                value = "version",
                required = true
            )
        }) Map namedArgs
    ) {
        throw new IllegalStateException("Groovy did not generate MinecraftExtension.mappings(Map)");
    }

    /// Sets the AccessTransformer configuration to use.
    ///
    /// @param configFile The configuration file to use
    void setAccessTransformer(RegularFile configFile);

    /// Sets the AccessTransformer configuration to use.
    ///
    /// @param configFile The configuration file to use
    void setAccessTransformer(File configFile);

    /// Sets the AccessTransformer configuration to use.
    ///
    /// The given object is resolved using [org.gradle.api.Project#file(Object)].
    ///
    /// @param configFile The configuration file to use
    void setAccessTransformer(Object configFile);

    /// Sets the AccessTransformer configuration to use.
    ///
    /// If the given provider does not provide a [file][File] or [regular file][RegularFile], the result will be
    /// resolved using [org.gradle.api.Project#file(Object)], similarly to [#setAccessTransformer(Object)].
    ///
    /// @param configFile The configuration file to use
    void setAccessTransformer(Provider<?> configFile);

    /// Configures the AccessTransformer options for this project.
    ///
    /// @param options The options to apply
    default void accessTransformer(Action<? super AccessTransformersContainer.Options> options) {
        this.accessTransformer(Closures.action(this, options));
    }

    /// Configures the AccessTransformer options for this project.
    ///
    /// @param options The options to apply
    @SuppressWarnings("rawtypes") // public-facing closure
    void accessTransformer(
        @DelegatesTo(value = AccessTransformersContainer.Options.class, strategy = Closure.DELEGATE_FIRST)
        @ClosureParams(value = SimpleType.class, options = "net.minecraftforge.accesstransformers.gradle.AccessTransformersContainer.Options")
        Closure options
    );


    /* DEPENDENCY OVERRIDES */

    @Override
    default boolean isChanging() {
        return false;
    }

    @Override
    default MinecraftDependency setChanging(boolean changing) {
        return this;
    }

    @Override
    MinecraftDependency copy();
}
