/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */
package net.minecraftforge.gradle;

import groovy.lang.Closure;
import groovy.lang.DelegatesTo;
import groovy.transform.Generated;
import groovy.transform.NamedParam;
import groovy.transform.NamedParams;
import groovy.transform.NamedVariant;
import groovy.transform.stc.ClosureParams;
import groovy.transform.stc.SimpleType;
import net.minecraftforge.accesstransformers.gradle.AccessTransformersContainer;
import org.gradle.api.Action;
import org.gradle.api.NamedDomainObjectContainer;
import org.gradle.api.artifacts.dsl.RepositoryHandler;
import org.gradle.api.artifacts.repositories.MavenArtifactRepository;
import org.gradle.api.attributes.Attribute;
import org.gradle.api.file.RegularFile;
import org.gradle.api.provider.Provider;
import org.gradle.api.provider.ProviderConvertible;
import org.gradle.nativeplatform.OperatingSystemFamily;
import org.jetbrains.annotations.Contract;
import org.jetbrains.annotations.UnknownNullability;

import java.io.File;
import java.io.Serializable;
import java.util.Map;
import java.util.Objects;

/// The main extension for ForgeGradle, where the Minecraft dependency resolution takes place.
///
/// ## Restrictions
///
/// - When declaring Minecraft dependencies, only [module
/// dependencies](https://docs.gradle.org/current/userguide/declaring_dependencies.html#1_module_dependencies) are
/// supported.
///   - The resulting Minecraft dependency is created by the Minecraft Mavenizer. It is not merely a dependency
/// transformation, which means that it cannot use file and project dependencies to generate the Minecraft artifacts.
///   - Attempting to provide a non-module dependency to [MinecraftExtension.ForProject#dep(Object)], will cause the
/// build to fail.
@SuppressWarnings("unused") // TODO [ForgeGradle7][Testing] Write functional tests
public sealed interface MinecraftExtension permits MinecraftExtensionImpl, MinecraftExtension.ForProject {
    /// The name for this extension in Gradle.
    String NAME = "minecraft";

    /**
     * A closure for the generated Minecraft maven to be passed into
     * {@link org.gradle.api.artifacts.dsl.RepositoryHandler#maven(Closure)}.
     * <p>Declaring this in your buildscript is <strong>required</strong> for the Minecraft dependencies to resolve
     * properly.</p>
     * <pre><code>
     * repositories {
     *     maven minecraft.maven
     * }
     * </code></pre>
     *
     * @return The closure
     */
    @SuppressWarnings("rawtypes") // public-facing closure
    Closure getMaven();

    /**
     * Adds the generated Minecraft maven to the given repository handler.
     * <pre><code>
     * minecraft.maven(repositories)
     * </code></pre>
     *
     * @param repositories The repository handler to add the maven to
     * @return The Minecraft maven
     * @see #getMaven()
     */
    default MavenArtifactRepository maven(RepositoryHandler repositories) {
        return repositories.maven(this.getMaven());
    }

    /**
     * The current mappings that were defined in the buildscript.
     *
     * @return The current mappings
     * @throws NullPointerException If no mappings were declared
     * @see #mappings(String, String)
     */
    Mappings getMappings();

    /**
     * Sets the mappings to use for the Minecraft Maven.
     * <p>This method includes a generated named variant that can make declaration in your buildscript easier.</p>
     * <pre><code>
     * minecraft {
     *     mappings channel: 'official', version: '1.21.5'
     * }
     * </code></pre>
     *
     * @param channel The mappings channel
     * @param version The mappings version
     * @throws IllegalArgumentException If any parameter is {@code null}
     * @apiNote Mappings should only be declared once. A warning will be reported on re-declaration.
     * @see <a href="https://docs.groovy-lang.org/latest/html/api/groovy/transform/NamedVariant.html">NamedVariant</a>
     */
    @NamedVariant
    void mappings(String channel, String version);

    /// Sets the mappings to use for the Minecraft Maven.
    ///
    /// This method is generated by Groovy in the implementing class, but has been included here for convenience and IDE
    /// support.
    ///
    /// @param namedArgs The named arguments
    /// @throws IllegalArgumentException If any parameter is `null`
    /// @apiNote Mappings should only be declared once. A warning will be reported on re-declaration.
    /// @see #mappings(String, String)
    @Contract // empty contract so IDEs don't think this method always fails
    @Generated
    @SuppressWarnings("rawtypes")
    default void mappings(
        @NamedParams({
            @NamedParam(
                type = String.class,
                value = "channel",
                required = true
            ),
            @NamedParam(
                type = String.class,
                value = "version",
                required = true
            )
        }) Map namedArgs
    ) {
        throw new IllegalStateException("Groovy did not generate MinecraftExtension.mappings(Map)");
    }

    /// [Project][org.gradle.api.Project]-specific additions for the Minecraft extension. These will be accessible from
    /// the `minecraft` DSL object within your project's buildscript.
    ///
    /// @see MinecraftExtension
    sealed interface ForProject extends MinecraftExtension permits MinecraftExtensionImpl.ForProjectImpl {
        /// Sets the AccessTransformer configuration to use.
        ///
        /// @param configFile The configuration file to use
        void setAccessTransformer(RegularFile configFile);

        /// Sets the AccessTransformer configuration to use.
        ///
        /// @param configFile The configuration file to use
        void setAccessTransformer(File configFile);

        /// Sets the AccessTransformer configuration to use.
        ///
        /// The given object is resolved using [org.gradle.api.Project#file(Object)].
        ///
        /// @param configFile The configuration file to use
        void setAccessTransformer(Object configFile);

        /// Sets the AccessTransformer configuration to use.
        ///
        /// If the given provider does not provide a [file][File] or [regular file][RegularFile], the result will be
        /// resolved using [org.gradle.api.Project#file(Object)], similarly to [#setAccessTransformer(Object)].
        ///
        /// @param configFile The configuration file to use
        void setAccessTransformer(Provider<?> configFile);

        /// Configures the AccessTransformer options for this project.
        ///
        /// @param options The options to apply
        default void accessTransformer(Action<? super AccessTransformersContainer.Options> options) {
            this.accessTransformer(Closures.action(this, options));
        }

        /// Configures the AccessTransformer options for this project.
        ///
        /// @param options The options to apply
        @SuppressWarnings("rawtypes") // public-facing closure
        void accessTransformer(
            @DelegatesTo(value = AccessTransformersContainer.Options.class, strategy = Closure.DELEGATE_FIRST)
            @ClosureParams(value = SimpleType.class, options = "net.minecraftforge.accesstransformers.gradle.AccessTransformersContainer.Options")
            Closure options
        );

        /// The collection of Slime Launcher options with which to create the launcher tasks.
        ///
        /// @return The collection of run task options
        NamedDomainObjectContainer<SlimeLauncherOptions> getRuns();

        /// Configures the Slime Launcher options for this project, which will be used to create the launcher tasks.
        ///
        /// @param closure The configuring closure
        void runs(
            @DelegatesTo(NamedDomainObjectContainer.class)
            @ClosureParams(value = SimpleType.class, options = "org.gradle.api.NamedDomainObjectContainer<net.minecraftforge.gradle.SlimeLauncherOptions>")
            Closure<Void> closure
        );

        /// Configures the Slime Launcher options for this project, which will be used to create the launcher tasks.
        ///
        /// @param action The configuring action
        default void runs(Action<? super NamedDomainObjectContainer<SlimeLauncherOptions>> action) {
            this.runs(Closures.action(this, action));
        }

        /// Creates (or marks if existing) the given dependency as a Minecraft dependency and configures it with the
        /// given closure.
        ///
        /// @param value   The dependency
        /// @param closure The closure to configure the dependency with
        /// @return The dependency
        /// @see <a href="https://docs.gradle.org/current/userguide/declaring_dependencies.html">Declaring Dependencies
        /// in Gradle</a>
        @SuppressWarnings("rawtypes") // public-facing closure
        MinecraftDependency dep(
            Object value,
            @DelegatesTo(value = MinecraftDependency.class, strategy = Closure.DELEGATE_FIRST)
            @ClosureParams(value = SimpleType.class, options = "net.minecraftforge.gradle.MinecraftDependency")
            Closure closure
        );

        /// Creates (or marks if existing) the given dependency as a Minecraft dependency and applies the given action
        /// to it.
        ///
        /// @param value  The dependency
        /// @param action The action to apply to the dependency attributes
        /// @return The dependency
        /// @see <a href="https://docs.gradle.org/current/userguide/declaring_dependencies.html">Declaring Dependencies
        /// in Gradle</a>
        default MinecraftDependency dep(Object value, Action<? super MinecraftDependency> action) {
            return this.dep(value, Closures.action(this, action));
        }

        /// Creates (or marks if existing) the given dependency as a Minecraft dependency.
        ///
        /// @param value The dependency
        /// @return The dependency
        /// @see <a href="https://docs.gradle.org/current/userguide/declaring_dependencies.html">Declaring Dependencies
        /// in Gradle</a>
        default MinecraftDependency dep(Object value) {
            return this.dep(value, Closures.empty(this));
        }

        /// Creates (or marks if existing) the given dependency as a Minecraft dependency and configures it with the
        /// given closure.
        ///
        /// @param value   The dependency
        /// @param closure The closure to configure the dependency with
        /// @return The dependency
        /// @see <a href="https://docs.gradle.org/current/userguide/declaring_dependencies.html">Declaring Dependencies
        /// in Gradle</a>
        @SuppressWarnings("rawtypes") // public-facing closure
        default MinecraftDependency dep(
            Provider<?> value,
            @DelegatesTo(value = MinecraftDependency.class, strategy = Closure.DELEGATE_FIRST)
            @ClosureParams(value = SimpleType.class, options = "net.minecraftforge.gradle.MinecraftDependency")
            Closure closure
        ) {
            return this.dep(value.get(), closure);
        }

        /// Creates (or marks if existing) the given dependency as a Minecraft dependency and applies the given action
        /// to it.
        ///
        /// @param value  The dependency
        /// @param action The action to apply to the dependency attributes
        /// @return The dependency
        /// @see <a href="https://docs.gradle.org/current/userguide/declaring_dependencies.html">Declaring Dependencies
        /// in Gradle</a>
        default MinecraftDependency dep(Provider<?> value, Action<? super MinecraftDependency> action) {
            return this.dep(value, Closures.action(this, action));
        }

        /// Creates (or marks if existing) the given dependency as a Minecraft dependency.
        ///
        /// @param value The dependency
        /// @return The dependency
        /// @see <a href="https://docs.gradle.org/current/userguide/declaring_dependencies.html">Declaring Dependencies
        /// in Gradle</a>
        default MinecraftDependency dep(Provider<?> value) {
            return this.dep(value, Closures.empty(this));
        }

        /// Creates (or marks if existing) the given dependency as a Minecraft dependency and configures it with the
        /// given closure.
        ///
        /// @param value   The dependency
        /// @param closure The closure to configure the dependency with
        /// @return The dependency
        /// @see <a href="https://docs.gradle.org/current/userguide/declaring_dependencies.html">Declaring Dependencies
        /// in Gradle</a>
        @SuppressWarnings("rawtypes") // public-facing closure
        default MinecraftDependency dep(
            ProviderConvertible<?> value,
            @DelegatesTo(value = MinecraftDependency.class, strategy = Closure.DELEGATE_FIRST)
            @ClosureParams(value = SimpleType.class, options = "net.minecraftforge.gradle.MinecraftDependency")
            Closure closure
        ) {
            return this.dep(value.asProvider(), closure);
        }

        /// Creates (or marks if existing) the given dependency as a Minecraft dependency and applies the given action
        /// to it.
        ///
        /// @param value  The dependency
        /// @param action The action to apply to the dependency attributes
        /// @return The dependency
        /// @see <a href="https://docs.gradle.org/current/userguide/declaring_dependencies.html">Declaring Dependencies
        /// in Gradle</a>
        default MinecraftDependency dep(ProviderConvertible<?> value, Action<? super MinecraftDependency> action) {
            return this.dep(value, Closures.action(this, action));
        }

        /// Creates (or marks if existing) the given dependency as a Minecraft dependency.
        ///
        /// @param value The dependency
        /// @return The dependency
        /// @see <a href="https://docs.gradle.org/current/userguide/declaring_dependencies.html">Declaring Dependencies
        /// in Gradle</a>
        default MinecraftDependency dep(ProviderConvertible<?> value) {
            return this.dep(value, Closures.empty(this));
        }
    }

    /// The mappings used for the Minecraft dependency.
    ///
    /// @param channel The channel to use (i.e. `'official'`)
    /// @param version The version to use (i.e. `'1.21.5'`)
    /// @see ForProject#mappings(String, String)
    record Mappings(String channel, String version) implements Serializable {
        /// Creates a new Mappings object.
        ///
        /// Both the channel and version are required, and cannot be null.
        ///
        /// @param channel The channel to use
        /// @param version The version to use
        /// @throws NullPointerException If any parameter is `null`
        public Mappings(String channel, String version) {
            this.channel = Objects.requireNonNull(channel, "Mappings channel cannot be null");
            this.version = Objects.requireNonNull(version, "Mappings version cannot be null");
        }

        static void checkParam(ForgeGradleProblems problems, @UnknownNullability Object param, String name) {
            if (param == null)
                throw problems.nullMappingsParam(name);
        }
    }

    /**
     * The attributes object for easy reference.
     * <pre><code>
     * dependencies {
     *     implementation 'com.example:example:1.0' {
     *         attributes.attribute(minecraft.attributes.os, objects.named(OperatingSystemFamily, OperatingSystemFamily.WINDOWS))
     *     }
     * }
     * </code></pre>
     *
     * @see Attributes
     */
    Attributes attributes = new MinecraftExtensionImpl.AttributesImpl();

    /// This interface contains the attributes used by the [Minecraft][MinecraftExtension] extension for resolving the
    /// Minecraft and deobfuscated dependencies.
    ///
    /// @see MinecraftExtension#attributes
    sealed interface Attributes permits MinecraftExtensionImpl.AttributesImpl {
        /// The [operating system family][OperatingSystemFamily] of the project's host.
        ///
        /// This is used to filter natives from the Minecraft repo.
        Attribute<OperatingSystemFamily> os = Attribute.of("net.minecraftforge.native.operatingSystem", OperatingSystemFamily.class);
        /// The requested mappings channel of the project.
        ///
        /// This is determined using [Mappings#channel()] via [#getMappings()]
        ///
        /// @see #mappingsVersion
        Attribute<String> mappingsChannel = Attribute.of("net.minecraftforge.mappings.channel", String.class);
        /// The requested mappings version of the project.
        ///
        /// This is determined using [Mappings#version()] via [#getMappings()]
        ///
        /// @see #mappingsChannel
        Attribute<String> mappingsVersion = Attribute.of("net.minecraftforge.mappings.version", String.class);
    }
}
