/*
 * Copyright (c) Forge Development LLC and contributors
 * SPDX-License-Identifier: LGPL-2.1-only
 */
package net.minecraftforge.forgedev.tasks.patching.diff

import groovy.transform.CompileStatic
import groovy.transform.PackageScope
import net.minecraftforge.forgedev.Tools
import net.minecraftforge.forgedev.tasks.ToolExec
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.file.FileSystemLocation
import org.gradle.api.file.FileSystemLocationProperty
import org.gradle.api.provider.Property
import org.gradle.api.tasks.Console
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.Optional

import javax.inject.Inject

@CompileStatic
@PackageScope abstract class BaseDiffPatchExec extends ToolExec {
    /* CLI FLAGS - See io.codechicken.diffpatch.cli.DiffPatchCli#mainI, or run --help on the fat jar */

    // Utility
    abstract @Input @Console Property<Boolean> getVerbose()
    abstract @Input @Optional @Console Property<String> getLogLevel()
    abstract @Input @Console Property<Boolean> getSummary()

    // Shared
    FileSystemLocationProperty<? extends FileSystemLocation> getInput() {
        throw new IllegalStateException('Must be overridden with RegularFileProperty or DirectoryProperty')
    }
    FileSystemLocationProperty<? extends FileSystemLocation> getOutput() {
        throw new IllegalStateException('Must be overridden with RegularFileProperty or DirectoryProperty')
    }
    abstract @Input @Optional Property<String> getArchive()
    abstract @Input @Optional Property<String> getArchiveBase()
    abstract @Input @Optional Property<String> getBasePathPrefix()
    abstract @Input @Optional Property<String> getModifiedPathPrefix()
    abstract @Input @Optional Property<String> getLineEndings()

    @Inject
    BaseDiffPatchExec() {
        super(Tools.DIFFPATCH)

        if (this.input instanceof DirectoryProperty)
            this.archiveBase.unset().disallowChanges()
        if (this.output instanceof DirectoryProperty)
            this.archive.unset().disallowChanges()

        this.verbose.convention(false)
        this.summary.convention(false)
    }

    protected void addArguments() {
        //region Utility
        if (this.verbose.get())
            this.args('--verbose')
        if (this.logLevel.present)
            this.args('--log-level', this.logLevel.get())
        if (this.summary.get())
            this.args('--summary')
        //endregion

        //region Shared
        if (this.output.present)
            this.args('--output', this.output.locationOnly.map(this.problems.ensureFileLocation()).get().asFile.absolutePath)
        if (this.archive.present)
            this.args('--archive', this.archive.get())
        if (this.archiveBase.present)
            this.args('--archive-base', this.archiveBase.get())
        if (this.basePathPrefix.present)
            this.args('--base-path-prefix', this.basePathPrefix.get())
        if (this.modifiedPathPrefix.present)
            this.args('--modified-path-prefix', this.modifiedPathPrefix.get())
        if (this.lineEndings.present)
            this.args('--line-endings', this.lineEndings.map {
                switch (it) {
                    case '\r':
                        return 'CR'
                    case '\n':
                        return 'LF'
                    case '\r\n':
                        return 'CRLF'
                    default:
                        return it
                }
            }.get())
        //endregion

        super.addArguments()
    }
}
