/*
 * Copyright (c) Forge Development LLC
 * SPDX-License-Identifier: LGPL-2.1-only
 */
package net.minecraftforge.gitversion.gradle.changelog

import groovy.transform.CompileStatic
import groovy.transform.PackageScope
import groovy.transform.PackageScopeTarget
import org.gradle.api.Project
import org.gradle.api.Task
import org.gradle.api.file.FileCollection
import org.gradle.api.publish.PublishingExtension
import org.gradle.api.publish.maven.MavenArtifact
import org.gradle.api.publish.maven.MavenPublication
import org.gradle.api.tasks.TaskOutputs
import org.gradle.api.tasks.TaskProvider
import org.gradle.language.base.plugins.LifecycleBasePlugin
import org.jetbrains.annotations.Nullable

/** Utility methods for configuring and working with the changelog tasks. */
@CompileStatic
@PackageScope([PackageScopeTarget.CLASS, PackageScopeTarget.CONSTRUCTORS, PackageScopeTarget.METHODS])
class ChangelogUtils {
    /**
     * Adds the createChangelog task to the target project. Also exposes it as a artifact of the 'createChangelog'
     * configuration.
     * <p>
     * This is the
     * <a href="https://docs.gradle.org/current/samples/sample_cross_project_output_sharing.html"> recommended way</a>
     * to share task outputs between multiple projects.
     *
     * @param project Project to add the task to
     * @return The task responsible for generating the changelog
     */
    static TaskProvider<? extends GenerateChangelog> setupChangelogTask(Project project) {
        project.tasks.register(GenerateChangelog.NAME, GenerateChangelogImpl).tap { task ->
            project.configurations.register(GenerateChangelog.NAME) { it.canBeResolved = false }
            project.artifacts.add(GenerateChangelog.NAME, task)
            project.tasks.named(LifecycleBasePlugin.ASSEMBLE_TASK_NAME).configure { it.dependsOn task }
        }
    }

    /**
     * Sets up the changelog generation on all maven publications in the project.
     * <p>It also sets up publishing for all subprojects as long as that subproject does not have another changelog plugin
     * overriding the propagation.</p>
     *
     * @param project The project to add changelog generation publishing to
     */
    static void setupChangelogGenerationOnAllPublishTasks(Project project) {
        setupChangelogGenerationForAllPublications(project)

        project.subprojects {
            Util.ensureAfterEvaluate(it) { subproject ->
                // attempt to get the current subproject's changelog extension
                var changelog = subproject.extensions.findByType(ChangelogExtension)

                // find the changelog extension for the highest project that has it, if the subproject doesn't
                for (var parent = project; changelog === null && parent !== null; parent = parent.parent == parent ? null : parent.parent) {
                    changelog = parent.extensions.findByType(ChangelogExtension)
                }

                // if the project with changelog is publishing all changelogs, set up changelogs for the subproject
                if (changelog?.publishAll?.getOrElse(false))
                    setupChangelogGenerationForAllPublications(subproject)
            }
        }
    }

    private static void setupChangelogGenerationForAllPublications(Project project) {
        var publishing = project.extensions.findByName(PublishingExtension.NAME) as PublishingExtension
        if (publishing === null) return

        // Get each extension and add the publishing task as a publishing artifact
        publishing.publications.withType(MavenPublication).configureEach { publication ->
            setupChangelogGenerationForPublishing(project, publication)
        }
    }

    private static ChangelogExtensionInternal findParent(Project project) {
        var changelog = project.extensions.findByType(ChangelogExtension) as ChangelogExtensionInternal
        if (changelog?.generating) return changelog

        var parent = project.parent == project ? null : project.parent
        return parent === null ? null : findParent(parent)
    }

    /**
     * The recommended way to share task outputs across projects is to export them as dependencies
     * <p>
     * So for any project that doesn't generate the changelog directly, we must create a
     * {@linkplain CopyChangelog copy task} and new configuration
     */
    private static @Nullable TaskProvider<? extends Task> findChangelogTask(Project project) {
        // See if we've already made the task
        if (project.tasks.names.contains(GenerateChangelog.NAME))
            return project.tasks.named(GenerateChangelog.NAME)

        if (project.tasks.names.contains(CopyChangelog.NAME))
            return project.tasks.named(CopyChangelog.NAME)

        // See if there is any parent with a changelog configured
        findParent(project)?.copyTo(project)
    }

    /**
     * Sets up the changelog generation on the given maven publication.
     *
     * @param project The project in question
     * @param publication The publication in question
     */
    static void setupChangelogGenerationForPublishing(Project project, MavenPublication publication) {
        Util.ensureAfterEvaluate(project) { p ->
            setupChangelogGenerationForPublishingAfterEvaluation(p, publication)
        }
    }

    private static void setupChangelogGenerationForPublishingAfterEvaluation(Project project, MavenPublication publication) {
        boolean existing = !publication.artifacts.findAll { MavenArtifact a -> a.classifier == 'changelog' && a.extension == 'txt' }.isEmpty()
        if (existing) return

        // Grab the task
        var task = findChangelogTask(project)

        // Add a new changelog artifact and publish it
        publication.artifact(task.map { it.outputs.files.singleFile }) { artifact ->
            artifact.builtBy(task)
            artifact.classifier = 'changelog'
            artifact.extension = 'txt'
        }
    }
}
