/*
 * Copyright (c) Forge Development LLC
 * SPDX-License-Identifier: LGPL-2.1-only
 */
package net.minecraftforge.testing.aggregate

import groovy.transform.CompileDynamic
import groovy.transform.CompileStatic
import groovy.xml.DOMBuilder
import groovy.xml.dom.DOMCategory
import org.gradle.api.DefaultTask
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.file.ProjectLayout
import org.gradle.api.file.RegularFileProperty
import org.gradle.api.model.ObjectFactory
import org.gradle.api.tasks.InputDirectory
import org.gradle.api.tasks.OutputFile
import org.gradle.api.tasks.TaskAction

import javax.inject.Inject

@CompileStatic
abstract class AggregateTestImpl extends DefaultTask implements AggregateTestInternal {
    protected abstract @InputDirectory DirectoryProperty getInput();
    @Override abstract @OutputFile RegularFileProperty getOutput();

    private final AggregateTestProblems problems = objects.newInstance(AggregateTestProblems)
    protected abstract @Inject ObjectFactory getObjects();
    protected abstract @Inject ProjectLayout getLayout();

    @Inject
    AggregateTestImpl() {
        this.input.convention(this.layout.buildDirectory.dir(TEST_RESULTS_DIRECTORY).map(problems.ensureFileLocation()))
        this.output.convention(this.layout.projectDirectory.file('aggregate-test-results.html'))
    }

    @TaskAction
    @CompileDynamic
    protected void exec() throws IOException {
        final javas = [:] as TreeMap
        final results = [:] as TreeMap

        for (def java : this.input.asFile.get().listFiles()) {
            for (def test : java.listFiles()) {
                if (!test.name.startsWith('TEST-') || !test.name.endsWith('.xml'))
                    continue

                def dirName = test.parentFile.name
                def (javaName, javaVersion) = dirName.split('-')
                javas.computeIfAbsent(javaName, { [] as SortedSet }).add(javaVersion)

                def data = DOMBuilder.parse(new StringReader(test.text)).documentElement
                use(DOMCategory) {
                    def suite = data['@name']
                    suite = suite.substring(suite.lastIndexOf('.') + 1)
                    def byTest = results.computeIfAbsent(suite, { [:] })
                    for (def testcase : data.testcase) {
                        def name = testcase['@name']
                        if (name.startsWith('repetition '))
                            continue
                        def byJava = byTest.computeIfAbsent(name, { [:] })
                        def byVersion = byJava.computeIfAbsent(javaName, { [:] })
                        byVersion.put(javaVersion, testcase.failure.isEmpty())
                    }
                }
                test.delete()
            }
            if (java.listFiles().length == 0)
                java.deleteDir()
        }

        def buffer = new StringBuilder()
        buffer.append("""
<html>
  <style>
    .tooltip-text {
      visibility: hidden;
      position: absolute;
      z-index: 1;
      width: 100px;
      color: white;
      font-size: 12px;
      background-color: #192733;
      border-radius: 10px;
      padding: 10px 15px 10px 15px;
      top: -40px;
      left: -50%;
    }

    .hover-text:hover .tooltip-text {
      visibility: visible;
    }

    .success {
      background-color: #008000;
    }

    .failure {
      background-color: #b60808;
    }

    .hover-text {
      font-size: 16px;
      position: relative;
      display: inline;
      font-family: monospace;
      text-align: center;
    }

    table, th, td {
      border: 1px solid black;
      border-collapse: collapse;
    }

    th, td {
       padding-left: 3px;
       padding-right: 3px;
    }

    .result {
        font-size: 0px;
    }
  </style>
  <body>
""")
        results.forEach { suite, byTest ->
            buffer << "<h1>$suite</h1>\n"
            buffer << "<table>\n"
            buffer << "  <thead>\n"
            buffer << "    <th>Test</th>\n"
            javas.keySet().forEach { javaName ->
                buffer << "    <th>$javaName</th>\n"
            }
            buffer << "  </thead>\n"
            buffer << "  <tbody>\n"
            byTest.forEach { testName, byJava ->
                buffer << "    <tr>\n"
                buffer << "      <td>$testName</td>\n"
                javas.forEach { javaName, versions ->
                    buffer << "      <td class=\"result\">\n"
                    def byVersion = byJava.get(javaName)
                    versions.forEach { ver ->
                        if (byVersion.containsKey(ver) && byVersion.get(ver)) {
                            buffer << "        <span class=\"hover-text success\">O<span class=\"tooltip-text success\" id=\"failure\">$javaName v$ver</span></span>\n"
                        } else {
                            buffer << "        <span class=\"hover-text failure\">X<span class=\"tooltip-text failure\">$javaName v$ver</span></span>\n"
                        }
                    }
                    buffer << "      </td>\n"
                }
                buffer << "    </tr>\n"
            }
            buffer << "  </tbody>\n"
            buffer << "</table>\n"

        }
        buffer << "</body></html>"

        this.output.asFile.get().text = buffer
    }
}
